package ge.id.plugin;

import org.jmrtd.protocol.SecureMessagingWrapper;

import javax.smartcardio.CardChannel;
import javax.smartcardio.CardException;
import javax.smartcardio.CommandAPDU;
import javax.smartcardio.ResponseAPDU;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

public class CommonUtils {

    private static final int FILE_PORTION_LENGTH = 234;

    public static int checkSuccess(ResponseAPDU resp) throws CardException {
        if (resp.getSW1() == 0x61) {
            int sw2 = resp.getSW2();
            return sw2 == 0x00 ? 256 : sw2;
        }

        if (resp.getSW() != 0x9000) {
            throw new CardException("Card Returned non-ok response. Response: " + Integer.toHexString(resp.getSW()));
        }
        return 0;
    }

    public static void select(CardChannel channel, SecureMessagingWrapper wrapper, int p1, int p2, int shortId) throws IOException, CardException {
        byte[] dfIdBytes;
        try (ByteArrayOutputStream bout = new ByteArrayOutputStream();
             DataOutputStream str = new DataOutputStream(bout)) {
            str.writeShort(shortId);
            dfIdBytes = bout.toByteArray();
        }
        select(channel, wrapper, p1, p2, dfIdBytes);
    }

    public static void select(CardChannel channel, SecureMessagingWrapper wrapper, int p1, int p2, byte[] shortId) throws CardException {
        CommandAPDU command = new CommandAPDU(0x00, 0xA4, p1, p2, shortId);
        ResponseAPDU response;
        if (wrapper != null) {
            response = sendWrappedApdu(channel, wrapper, command);
        } else {
            response = channel.transmit(command);
        }
        checkSuccess(response);
    }

    public static void selectDF(CardChannel channel, int shortId) throws IOException, CardException {
        select(channel, null, 0x01, 0x04, shortId);
    }

    public static void selectDF(CardChannel channel, byte[] dfAid) throws CardException {
        select(channel, null, 0x04, 0x0C, dfAid);
    }

    public static void selectFile(CardChannel channel, int shortId) throws IOException, CardException {
        selectFile(channel, null, shortId);
    }

    public static void selectFile(CardChannel channel, SecureMessagingWrapper wrapper, int shortId) throws IOException, CardException {
        select(channel, wrapper, 0x09, 0x04, shortId);
    }

    public static void selectFile(CardChannel channel, byte[] shortId) throws CardException {
        select(channel, null, 0x09, 0x04, shortId);
    }

    public static void selectAID(CardChannel channel, byte[] aid) throws CardException {
        select(channel, null, 0x04, 0x00, aid);
    }

    public static byte[] getData(CardChannel channel, int shortId) throws CardException {
        int p1 = ((shortId >> 8) & 0xff);
        int p2 = (shortId & 0xff);
        CommandAPDU commandAPDU = new CommandAPDU(0x00, 0xCA, p1, p2, 256);
        ResponseAPDU resp = channel.transmit(commandAPDU);
        if (resp.getSW1() == 0x6C) {
            commandAPDU = new CommandAPDU(0x00, 0xCA, p1, p2, resp.getSW2());
            resp = channel.transmit(commandAPDU);
        }
        checkSuccess(resp);
        return resp.getData();
    }

    public static ResponseAPDU checkSuccessAndGetRemainingData(CardChannel channel, ResponseAPDU response)
            throws CardException, IOException {

        try (ByteArrayOutputStream out = new ByteArrayOutputStream(); DataOutputStream d = new DataOutputStream(out)) {
            int remainingResponse = checkSuccess(response);
            out.write(response.getData());
            while (remainingResponse > 0) {
                response = getResponse(channel, remainingResponse);
                remainingResponse = checkSuccess(response);
                out.write(response.getData());
            }

            d.writeShort(response.getSW());
            return new ResponseAPDU(out.toByteArray());
        }
    }

    public static ResponseAPDU getResponse(CardChannel channel, int dataLength) throws CardException {
        return channel.transmit(new CommandAPDU(0x00, 0xC0, 0x00, 0x00, dataLength));
    }

    public static byte[] getFullResponse(CardChannel channel) throws CardException, IOException {
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
            int remainingResponse = 256;
            do {
                ResponseAPDU resp = getResponse(channel, remainingResponse);
                remainingResponse = checkSuccess(resp);
                out.write(resp.getData());
            } while (remainingResponse > 0);

            return out.toByteArray();
        }

    }

    static class FileChunk {
        byte[] data;
        int offset;
        boolean lastOne = false;
    }

    public static byte[] calculateFileOffset(int offset) {
        if (offset < 0 || offset > 0x7FFF) {
            throw new IllegalArgumentException(
                    "File offset must be in [0x0000 - 0x7FFF] segment, not 0x" + Integer.toHexString(offset));
        }

        byte[] buf = new byte[2];
        buf[0] = (byte) ((offset & 0xff00) >> 8);
        buf[1] = (byte) (offset & 0xff);
        return buf;
    }


    public static byte[] readCurrentFile(CardChannel channel) throws CardException, IOException {

        ByteArrayOutputStream bos = new ByteArrayOutputStream();

        FileChunk chunk = new FileChunk();
        while (!chunk.lastOne) {
            byte[] p1p2Values = calculateFileOffset(chunk.offset);

            readFileChunk(chunk, p1p2Values[0], p1p2Values[1], channel);
            bos.write(chunk.data);

        }

        return bos.toByteArray();
    }

    private static void readFileChunk(FileChunk chunk, byte p1, byte p2, CardChannel basicChannel)
            throws CardException {
        int ne = 256;
        CommandAPDU cmd = new CommandAPDU(0x00, 0xB0, p1, p2, ne);

        ResponseAPDU resp = basicChannel.transmit(cmd);

        chunk.data = resp.getData();
        int dataLength = chunk.data.length;
        switch (resp.getSW()) {
            case 0x9000:
                chunk.offset += dataLength;
                break;
            case 0x6282:
            case 0x6B00:
                chunk.offset += dataLength;
                chunk.lastOne = true;
                break;
            default:
                throw new CardException(Integer.toHexString(resp.getSW()));

        }

    }

    public static long bytesToLong(byte[] bytes) {
        if (bytes.length > 8) {
            throw new IllegalArgumentException(
                    String.format("Bytes length is %d, and cannot be more than 8", bytes.length));
        }
        byte[] b = new byte[8];
        System.arraycopy(bytes, 0, b, 8 - bytes.length, bytes.length);
        ByteBuffer bb = ByteBuffer.wrap(b);
        bb.order(ByteOrder.BIG_ENDIAN);
        return bb.getLong();
    }

    public static ResponseAPDU sendWrappedApdu(CardChannel cardChannel, SecureMessagingWrapper wrapper, CommandAPDU cmd) throws CardException {
        return sendWrappedApdu(cardChannel, wrapper, cmd, true);
    }

    public static ResponseAPDU sendWrappedApdu(CardChannel cardChannel, SecureMessagingWrapper wrapper, CommandAPDU cmd, boolean unwrap) throws CardException {
        net.sf.scuba.smartcards.CommandAPDU scubaCmd = new net.sf.scuba.smartcards.CommandAPDU(cmd.getBytes());
        scubaCmd = wrapper.wrap(scubaCmd);
//        System.out.println("Request: " + Hex.toHexString(scubaCmd.getBytes()));
        ResponseAPDU responseApdu = cardChannel.transmit(new CommandAPDU(scubaCmd.getBytes()));
        net.sf.scuba.smartcards.ResponseAPDU scubaRsp = new net.sf.scuba.smartcards.ResponseAPDU(responseApdu.getBytes());
//        System.out.println("Response: " + Hex.toHexString(scubaRsp.getBytes()));
        if (unwrap) {
            scubaRsp = wrapper.unwrap(scubaRsp);
        }
        return new ResponseAPDU(scubaRsp.getBytes());
    }
}
