package ge.id.plugin;

import eu.europa.esig.dss.enumerations.DigestAlgorithm;
import eu.europa.esig.dss.enumerations.SignatureAlgorithm;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.SignatureValue;
import eu.europa.esig.dss.model.ToBeSigned;
import eu.europa.esig.dss.model.x509.CertificateToken;
import eu.europa.esig.dss.token.AbstractSignatureTokenConnection;
import eu.europa.esig.dss.token.DSSPrivateKeyEntry;
import eu.europa.esig.dss.token.PasswordInputCallback;
import lu.nowina.nexu.PinHolderForCard;
import lu.nowina.nexu.api.DetectedCard;
import lu.nowina.nexu.api.MessageDisplayCallback;
import lu.nowina.nexu.api.NexuAPI;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.Hex;

import javax.smartcardio.*;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.security.MessageDigest;
import java.security.Security;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class GeorgianStampCardSignatureTokenConnection extends AbstractSignatureTokenConnection {

    private NexuAPI api;
    private DetectedCard card;
    private PasswordInputCallback passwordCallback;
    private MessageDisplayCallback messageCallback;

    public GeorgianStampCardSignatureTokenConnection(
            NexuAPI api, DetectedCard card, PasswordInputCallback passwordCallback,
            MessageDisplayCallback messageCallback
    ) {
        Security.addProvider(new BouncyCastleProvider());
        this.api = api;
        this.card = card;
        this.passwordCallback = passwordCallback;
        this.messageCallback = messageCallback;
    }

    @Override
    public void close() {
    }

    @Override
    public synchronized List<DSSPrivateKeyEntry> getKeys() throws DSSException {
        // Should contain both sign and auth pairs
        List<DSSPrivateKeyEntry> entries = new ArrayList<>();

        try {
            // Get card terminal
            CardTerminal cardTerminal = api.getCardTerminal(card);
            Card terminalCard = cardTerminal.connect("*");

            CardChannel basicChannel = terminalCard.getBasicChannel();

            CommonUtils.selectDF(basicChannel, Constants.MF_ID);
            CommonUtils.selectDF(basicChannel, Constants.SSCD_ID);

            CommonUtils.selectFile(basicChannel, Constants.ESIG_CERT_CA_ID);
            byte[] certCABytes = CommonUtils.readCurrentFile(basicChannel);

            CommonUtils.selectFile(basicChannel, Constants.ESIG_CERT_ID);
            byte[] certBytes = CommonUtils.readCurrentFile(basicChannel);

            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            try (ByteArrayInputStream certBais = new ByteArrayInputStream(certBytes); //
                 ByteArrayInputStream caBais = new ByteArrayInputStream(certCABytes)) {
                X509Certificate cert = (X509Certificate) cf.generateCertificate(certBais);

                X509Certificate certCA = (X509Certificate) cf.generateCertificate(caBais);

                List<X509Certificate> signChain = Arrays.asList( //
                        cert, //
                        certCA);

                entries.add(new EidPrivateKeyEntry(new CertificateToken(signChain.get(0)), signChain));
            }
        } catch (Exception e) {
            throw new DSSException(e);
        }
        return entries;
    }

    @Override
    public SignatureValue sign(ToBeSigned toBeSigned, DigestAlgorithm digestAlgorithm, DSSPrivateKeyEntry keyEntry)
            throws DSSException {
        try {
            // Get card terminal
            CardTerminal cardTerminal = api.getCardTerminal(card);
            Card terminalCard = cardTerminal.connect("*");
            CardChannel basicChannel = terminalCard.getBasicChannel();

            CommandAPDU cmd;
            ResponseAPDU resp;

            CommonUtils.selectDF(basicChannel, Constants.MF_ID);
            CommonUtils.selectDF(basicChannel, Constants.SSCD_ID);

            // Check if we have this certificate saved
            char[] password = null;
            if (api.getAppConfig().isCachePin() //
                    && PinHolderForCard.NAME != null //
                    && PinHolderForCard.NAME.equals(keyEntry.getCertificate().getCertificate().getSubjectDN().toString())) {
                password = PinHolderForCard.PIN;

            } else {
                password = passwordCallback.getPassword();

                if (api.getAppConfig().isCachePin()) {
                    PinHolderForCard.NAME = keyEntry.getCertificate().getCertificate().getSubjectDN().toString();
                    PinHolderForCard.PIN = password;
                }
            }

            resp = basicChannel.transmit(PinPukHelper.verifyPin(0x85, String.valueOf(password), 0x0C));
            CommonUtils.checkSuccess(resp);

            cmd = new CommandAPDU(0x00, 0x22, 0x41, 0xB6, Hex.decode("84019E800142"));
            resp = basicChannel.transmit(cmd);
            CommonUtils.checkSuccess(resp);

            byte[] data = toBeSigned.getBytes();

            MessageDigest digest = MessageDigest.getInstance(digestAlgorithm.getName());

            byte[] toSign = digest.digest(data);

            cmd = new CommandAPDU(0x00, 0x2A, 0x9E, 0x9A, toSign);

            byte[] computeSignatureCmd;
            try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
                out.write(cmd.getBytes());
                out.write(0);
                computeSignatureCmd = out.toByteArray();
            }
            resp = basicChannel.transmit(new CommandAPDU(computeSignatureCmd));
            CommonUtils.checkSuccess(resp);

            byte[] signed = resp.getData();

            SignatureValue signatureValue = new SignatureValue(SignatureAlgorithm.RSA_SHA256, signed);
            return signatureValue;
        } catch (

                Exception e) {
            throw new DSSException(e);
        }
    }
}
